<?php

namespace App\Http\Controllers;

use App\AppStatus;
use App\Imports\CfmsDataImport;
use App\Imports\TitheImport;
use App\Membership;
use App\OfferingType;
use App\PaymentOption;
use App\TitheOfferingBase;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Throwable;
use Yajra\DataTables\Exceptions\Exception;

class TitheOfferingBaseController extends Controller
{
    /**
     * @throws Exception
     * @throws \Exception
     */
    public function index()
    {
        if (request()->ajax()) {
            $data = TitheOfferingBase::query()
                ->with(['user'])
                ->withCount('titheOfferings')
                ->withSum('titheOfferings', 'Amount')
                ->where('company_id', '=', auth()->user()->company_id);

            return datatables()->of($data)
                ->addIndexColumn()
                ->addColumn('action', fn(TitheOfferingBase $titheOfferingBase) => view('tithe-offering_base._actions', compact('titheOfferingBase')))
                ->rawColumns(['action'])
                ->make(true);


        }
        $payments = PaymentOption::query()->where(
            "CompanyId",
            "=",
            auth()->user()->company_id
        )->get();$types = OfferingType::query()->where("CompanyId", "=", auth()->user()->company_id)->get();
        $members = Membership::query()->where("company_id", "=", auth()->user()->company_id)->get();
        return view('tithe-offering_base.index', compact('payments','members','types'));
    }

    /**
     * @throws Throwable
     */
    public function manualUpload(Request $request)
    {
        $file = $request->file("file");

        DB::beginTransaction();
        $model = $this->saveBaseData(TitheOfferingBase::MANUAL_UPLOAD);
        $import = new TitheImport($request->input("modeId"), $model);

        Excel::import($import, $file);


        if ($import->errors->count() <= 0) {
            DB::commit();
            return back()->with('success', 'All good!');
        } else {
            DB::rollBack();
            return back()->withErrors($import->errors->toArray());
        }
    }

    /**
     * @throws Throwable
     */
    public function uploadCmfsData(Request $request)
    {
        $data = $request->validate([
            'file' => ['required', 'mimes:xls,xlsx'],
            'offering_type_id' => ['required', 'exists:offering_types,id']
        ]);

        $file = $request->file('file');

        DB::beginTransaction();
        $model = $this->saveBaseData(TitheOfferingBase::CFMS_UPLOAD);
        $import = new CfmsDataImport($data['offering_type_id'], $model);
        Excel::import($import, $file);

        if ($import->errors->count() <= 0) {
            DB::commit();
            return back()->with('success', 'All good! Data imported successfully');
        } else {
            DB::rollBack();
            return back()->with('error', 'Unexpected error occurred while importing data. Please try again later.');
        }
    }

    /**
     * @param $source
     * @return TitheOfferingBase|Model
     */
    public function saveBaseData($source, $status = AppStatus::PENDING): TitheOfferingBase|Model
    {
        return TitheOfferingBase::query()->create([
            'source' => $source,
            'user_id' => auth()->id(),
            'company_id' => auth()->user()->company_id,
            'status' => $status
        ]);
    }

    public function show(TitheOfferingBase $titheOfferingBase)
    {
        $titheOfferingBase->load(['user'])
            ->loadCount('titheOfferings')
            ->loadSum('titheOfferings', 'Amount');
        $payments = PaymentOption::query()->where("CompanyId", "=", auth()->user()->company_id)->get();
        $types = OfferingType::query()->where("CompanyId", "=", auth()->user()->company_id)->get();
        $members = Membership::query()->where("company_id", "=", auth()->user()->company_id)->get();


        return view('tithe-offering_base.show', compact('titheOfferingBase', 'payments', 'types', 'members'));
    }

    public function approve(TitheOfferingBase $offeringBase)
    {
        $offeringBase->status = AppStatus::APPROVED;
        $offeringBase->save();
        if (request()->ajax()) {
            session()->flash('success', 'Offering approved successfully');
            return response()->json([
                'success' => true,
                'message' => 'Offering approved successfully'
            ]);
        }
        return back()->with('success', 'Offering approved successfully');
    }

    /**
     * @throws Throwable
     */
    public function destroy(TitheOfferingBase $titheOfferingBase)
    {
        DB::beginTransaction();
        $titheOfferingBase->titheOfferings()->delete();
        $titheOfferingBase->delete();
        DB::commit();
        if (request()->ajax()) {
            session()->flash('success', 'Offering deleted successfully');
            return response()->json([
                'success' => true,
                'message' => 'Offering deleted successfully'
            ]);
        }
        return back()->with('success', 'Offering deleted successfully');
    }

}
