<?php

namespace App\Http\Controllers;

use App\AppStatus;
use App\Category;
use App\Exports\TitheOfferingExport;
use App\Exports\TitheTemplateExport;
use App\Imports\CfmsDataImport;
use App\Imports\MomoTitheImport;
use App\Imports\TitheImport;
use App\Imports\TithePreviewImport;
use App\Membership;
use App\OfferingType;
use App\PaymentOption;
use App\TitheAndOffering;
use App\TitheOfferingBase;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Facades\Excel;
use PhpOffice\PhpSpreadsheet\Exception;
use Yajra\DataTables\Facades\DataTables;

class TitheAndOfferingController extends Controller
{

    public function summaryReportExtract(Request $request)
    {
        $data = OfferingType::query()
            ->where("CompanyId", "=", auth()->user()->company_id)->get();
        return view(
            'tithe_and_offering.summary_report_extract',
            compact('data')
        )->with([
            'start' => $request->input("start-date"),
            'end' => $request->input("end-date"),
        ]);
    }

    public function detailedReportExtract(Request $request)
    {
        $start = $request->input("start-date");
        $end = $request->input("end-date");
        $data = TitheAndOffering::query()->where("Date", ">=", $start)
            ->where("Date", "<=", $end)
            ->where("CompanyId", "=", auth()->user()->company_id)
            ->get();
        return view(
            'tithe_and_offering.detailed_report_extract',
            compact('data')
        );
    }

    public function summaryReport()
    {
        return view('tithe_and_offering.summary_report');
    }

    public function detailedReport()
    {
        return view('tithe_and_offering.detailed_report');
    }

    public static function sumByDate(OfferingType $type, $start, $end)
    {
        return TitheAndOffering::query()->where("Date", ">=", $start)
            ->where("Date", "<=", $end)
            ->where("OfferingTypeId", $type->id)->sum('Amount');
    }

    /**
     * @throws \Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $start = $request->input("start_date");
            $end = $request->input("end_date");
            $data = TitheAndOffering::query()->with(["payment", "offering", 'member.smallgroups'])
                ->where("tithe_and_offerings.CompanyId", "=", auth()->user()->company_id)
                ->when($request->input("model_id"), function (Builder $builder) use ($request) {
                    $builder->where('tithe_offering_base_id', '=', $request->input("model_id"));
                })
                ->when($start && $end, function (Builder $builder) use ($end, $start) {
                    $builder->where("Date", ">=", $start)->where("Date", "<=", $end);
                })
                ->select("tithe_and_offerings.*");
            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('action', function ($row) {
                    return '<div class="btn-group btn-group-sm">
                                                        <a href="#" data-company="'
                        . $row->CompanyId . '"
                                                           data-id="' . $row->id . '"
                                                           data-member="'
                        . $row->MemberId . '"
                                                           data-offering="'
                        . $row->OfferingTypeId . '"
                                                           data-amount="'
                        . $row->Amount . '"
                                                           data-names="'
                        . htmlentities($row->member_names) . '"
                                                           data-phone="'
                        . htmlentities($row->member_phone) . '"
                                                           data-payment="'
                        . $row->PaymentOptionId . '"
                                                           data-date="'
                        . $row->Date . '"
                                                           class="btn btn-primary js-edit"><span class="fa fa-edit"></span> </a>
                                                        <a href="' . route(
                            "tithe.and.offering.delete",
                            $row->id
                        ) . '" class="btn btn-danger js-delete"><span
                                                                class="fa fa-trash"></span> </a>
                                                    </div>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        $payments = PaymentOption::query()->where("CompanyId", "=", auth()->user()->company_id)->get();
        $types = OfferingType::query()->where("CompanyId", "=", auth()->user()->company_id)->get();
        $members = Membership::query()->where("company_id", "=", auth()->user()->company_id)->get();


        $start = $request->input("start_date");
        $end = $request->input("end_date");

        if (!$start) {
            $start = "";
        }

        if (!$end) {
            $end = "";
        }

        return view(
            'tithe_and_offering.create',
            compact('payments', 'members', 'types', 'start', 'end')
        );
    }

    /**
     * @throws \Throwable
     */
    public function store(Request $request)
    {
        DB::beginTransaction();
        $model = app(TitheOfferingBaseController::class)->saveBaseData(TitheOfferingBase::MANUAL_ENTRY, AppStatus::APPROVED);

        $tithe = new TitheAndOffering();
        $tithe->tithe_offering_base_id = $model->id;
        $tithe->CompanyId = auth()->user()->company_id;
        $tithe->MemberId = $request->MemberId;
        $tithe->OfferingTypeId = $request->OfferingTypeId;
        $tithe->Amount = $request->Amount;
        $tithe->member_names = $request->member_names;
        $tithe->member_phone = $request->member_phone;
        $tithe->PaymentOptionId = $request->PaymentOptionId;
        $tithe->Date = $request->Date;
        $tithe->save();
        DB::commit();
        return redirect()->back()->with(
            ['success' => 'Tithe And Offering Created Successfully']
        );
    }

    public function update(Request $request)
    {
        $tithe = TitheAndOffering::FindOrFail($request->input('TitheId'));
        $tithe->OfferingTypeId = $request->OfferingTypeId;
        $tithe->Amount = $request->Amount;
        $tithe->member_names = $request->member_names;
        $tithe->member_phone = $request->member_phone;
        $tithe->PaymentOptionId = $request->PaymentOptionId;
        $tithe->Date = $request->Date;
//        dd($request->all());
        $tithe->save();
        return redirect()->back()->with(
            'success',
            'Tithe And Offering Updated Successfully'
        );
    }

    public function destroy($id)
    {
        $tithe = TitheAndOffering::find($id);
        $tithe->delete();
        return redirect()->back()
            ->with(
                'success',
                'Tithe And Offering Updated Successfully'
            );
    }

    /**
     * @throws Exception
     * @throws \PhpOffice\PhpSpreadsheet\Writer\Exception
     */
    public function excelTemplate()
    {
        return Excel::download(
            new TitheTemplateExport(),
            "Tithe And Offering Template " . time() . ".xlsx"
        );
    }

    public function uploadMomoFile(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv',
        ]);


        $import = new TithePreviewImport();
        $file = $request->file('file');
        Excel::import($import, $file);

        // Get the processed tithes data
        $tithes = $import->getTithes();


        // Group the data by category or perform other operations
        $groupedData = collect($tithes)->groupBy('offering_type')
            ->map(function ($rows, $category) {
                return [
                    'category' => $category,
                    'total_members' => $rows->count(),
                    'total_amount' => $rows->sum('amount'),
                ];
            });

        // Fetch the categories from the database for mapping
        $categories = OfferingType::all();

        $dir = 'uploads/tithes_preview';
        $path = $file->storeAs($dir, 'tithes_' . time() . '.' . $file->extension());
        session()->put('file_path', $path);
        return view('tithe_and_offering.momo_preview', compact('groupedData', 'categories', 'tithes'));
    }


    public function importTithes(Request $request)
    {
        $groupedCategories = collect($request->input('grouped_categories'));
        $categories = collect($request->input('categories'));
        $file = \Storage::path(session('file_path'));
        $import = new MomoTitheImport($groupedCategories, $categories);
        Excel::import($import, $file);

        // delete file from storage
        \Storage::delete(session('file_path'));
        // delete session data
        session()->forget('file_path');
        return redirect()->route('tithe.and.offering.create')->with('success', 'Tithes imported successfully');
    }

    /**
     * @throws Exception
     * @throws \PhpOffice\PhpSpreadsheet\Writer\Exception
     */
    public function exportToExcel()
    {
        $start = request('start_date');
        $end = request('end_date');
        $titheBaseId = request('model_id');
        return Excel::download(new TitheOfferingExport($start, $end, $titheBaseId), "tithes_{$start}_{$end}_.xlsx");
    }


}
